/**
 * \file: sdc_arch.h
 *
 * \brief : Definition of SDC function which have to be implemented in each
 * architecture dependent part
 *
 * <b>SDC private header</b>\n
 *
 * \author: Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/

/**
 * \defgroup arch Architecture specific API
 * Documentation of functions which need to be implemented for each architecture
 */

/**
 * \ingroup arch
 * \defgroup arch_gen Generic functions
 * Architecture specific functions not related to any operation
 */

/**
 * \ingroup arch
 * \defgroup arch_key Key management
 * Architecture specific functions not related to key management
 *
 * \todo add missing key import support (keystore key)
 *
 * In order to be able to import (to keystore) an wrapped key (with or without SDC daemon) the process is split in two steps.
 * - First the key is decrypted(unwrapped) using the current session key and afterwards reencrypted(wrapped) using a build-in key,
 * - second the key is imported to keystore after decrypting(unwrapping) using the same build-in key.
 * Functions:
 *  - sdc_arch_reencrypt_key - input session, encrypted key(buf + len), \ref sdc_wrap_unwrap_type_t for unwrap, IV(buf,len) for unwrap, build-in key spec (build-in key idx, any uid, permissions(incl. any gid)), \ref sdc_wrap_unwrap_type_t for wrap, IV(buf,len) for wrap, output wrapped key(buf,len) - the permissions are limited by the inheritance mask (see \ref sdc_permissions) of the key used for unwrapping
 *  - sdc_arch_import_storage_key - kid, kid_opt, reencypted_key(buf + len), key_stor_opt, permissions, build-in key spec (build-in key idx, any uid, permissions(incl. any gid)), \ref sdc_wrap_unwrap_type_t for unwrap, IV(buf,len) for unwrap - the permissions are limited by the inheritance mask (see \ref sdc_permissions) of the build-in key
 */

/**
 * \ingroup arch
 * \defgroup arch_session Session management
 * Architecture specific functions not related to session management
 *
 * \todo add missing key import support (session key)
 *
 * later we will add support to replace the current session key by an key wrapped with the current session key
 *  - sdc_arch_import_session_key - input session, wrapped key(buf + len), \ref sdc_wrap_unwrap_type_t, IV(buf,len) for unwrap, permissions - the permissions are limited by the inheritance mask (see \ref sdc_permissions) of the key used for unwrapping
 *
 * \todo add missing session only key support
 *
 * later we will add support to generate a random buffer (e.g. to be used as key) wrapped by the current session key
 *  - sdc_arch_generate_wrapped_random input session, desired length of random, \ref sdc_wrap_unwrap_type_t, IV(buf,len) for wrap, output wrapped random buf(buf,len)
 */

/**
 * \ingroup arch
 * \defgroup arch_keyinfo Key info
 * \todo add missing key info functions
 *
 * later we will add a function to return information of the key in the session or
 * any key referenced by kid
 *  - sdc_arch_storage_key_get_info - input kid, output filled key_info + next valid kid
 *  - sdc_arch_session_get_key_info - input session, output filled key_info
 */

/**
 * \ingroup arch
 * \defgroup arch_random Random
 * Architecture specific functions for random generation
 */

/**
 * \ingroup arch
 * \defgroup arch_encdec Encrypt/Decrypt
 * Architecture specific functions for encrypt/decrypt
 */

/**
 * \ingroup arch
 * \defgroup arch_signverify Sign/Verify
 * Architecture specific functions for sign/verify
 */

/**
 * \ingroup arch
 * \defgroup arch_digest Digest
 * Architecture specific functions for digest
 */

/**
 * \ingroup arch
 * \defgroup arch_wrapunwrap Wrap/Unwrap
 * Architecture specific functions for wrap/unwrap
 */

#ifndef __SDC_LIB_ARCH_H_
#define __SDC_LIB_ARCH_H_

#include <sdc.h>
#include <sdc_session.h>
#include <sdc_perm.h>
#include <sdc_keystore_keys.h>
#include <sdc_op_common.h>
#include <private/sdc_intern.h>

#include <sdc/arch/arch_session.h>

/**
  * \ingroup arch
  * \brief This Structure contains specific parameters for init/update/finalize of wrap/unwrap operation.
  *
  * This struct will be used to hold total input length, tag length and total aad length,
  * which will be used in init/update/finalize of wrap/unwrap operation
  */
 struct sdc_wrap_unwrap_param {
    size_t total_inlen; /**< total input length which is going to processed */
    size_t tag_len; /**< tag length  */
    size_t total_aadlen; /**< additional authenticated data for wrap/unwrap*/
 };

/**
  * \ingroup arch
  * \brief This Structure contains specific parameters for init/update/finalize of wrap/unwrap operation.
  */
 typedef struct sdc_wrap_unwrap_param sdc_wrap_unwrap_param_t;

/**
 * \ingroup arch
 * \brief Definition of the architecture specific session content
 *
 * Note: Architecture specific part must only be accessed from
 *       architecture specific code
 */
typedef struct sdc_arch_session sdc_arch_session_t;

/**
 * \ingroup arch
 * \brief Definition of the sdc_session struct
 *
 * Combine architecture and architecture independent part of the session.
 *
 * This struct will be used for \ref sdc_session_t (abstract type)
 * For this reason the content is only visible within libSDC.
 *
 * Note: Architecture specific part must only be accessed from
 *       architecture specific code
 */
struct sdc_session {
    /* Architecture independent part */
     uint8_t unaligned_buffer[SDC_ARCH_MAX_UNALIGNED_LEN]; /**< buffer to buffer potential unaligned part of the message till next update or finalize */
     size_t unaligned_buffer_filllevel; /**< number of bytes stored to \ref unaligned_buffer */
     size_t inlen_cnt; /**< counter to keep track on the amount of bytes processed so far - note it might be processed by HW or stored in unaligned_buffer */
     sdc_ops_sequence_t sequence; /**< operation sequence status parameters */
     sdc_wrap_unwrap_param_t iuf_wrap_unwrap; /**<wrap/unwrap specific parameters set during init - only valid for init-update-finalize */

     /* Architecture specific part */
     sdc_arch_session_t arch; /**< link \ref sdc_arch_session_t */
};

/* Definitions types and defaults */

/**
 * \struct sdc_session
 * \ingroup arch_session
 * \brief Architecture specific struct to collect all information of a session.
 *
 * Examples:
 *  - File descriptor used for the session
 *  - Currently selected kid
 *  - Buffers needed to handle unaligned requests
 */

/* Functions */

/**
 * \ingroup arch_key
 * \defgroup arch_keystore Keystore key
 * Architecture specific functions not related to management of keystore keys
 */

/**
 * \ingroup arch_keystore
 * \brief Architecture specific function to install a keystore key.
 *
 * Note: The system needs to guarantee that all persistent keys of the previous power cycle are restored
 * before any storage key can be installed or removed. While the keystore is being restored this function
 * needs to return \ref SDC_KEY_NOT_READY.
 *
 * Note: The owner of a keystore key needs to be the current effective uid
 *       The architecture specific implementation needs to enforce
 *       uid in permissions equals current effective uid
 *
 * The implementation of this function needs to return only after the key has been created and
 * in case of a persistent key stored to disk.
 * In case of a reboot during key installation it has to be guaranteed for persistent key that
 *  - the key is installed correctly (with correct settings, data etc.)
 *  - not available at all
 * Furthermore the installation of the key needs to guarantee internally that any (i.e. parallel running)
 * process can't reference or use the key (e.g. \ref sdc_session_load_storage_key) before the
 * key has created completely and in case of a persistent key been stored to disk.
 * This is required to guarantee, that even in case of reboot during key installation no data can
 * e.g. encrypted with a key which is not available after reboot.
 *
 * Note: the uid of the key is the effective uid of the requesting process.
 * Note: the gid of the key can be specified using \p permissions. Any gid can be used. Even
 * if the gid is neither the effective gid of the process nor in its supplementary groups.
 *
 * Note: in case the key ID is determined automatically \p kid_opt == \ref SDC_CREATE_KEY_AUTOMATIC_ID
 * the architecture dependent function needs to return a free id in the range of
 * \ref SDC_AUTOMATIC_KID_RANGE_MIN and \ref SDC_AUTOMATIC_KID_RANGE_MAX
 *
 * Please refer to \ref sdc_insert_storage_key for details on the parameters and return values.
 * The architecture independent wrapper will assure that
 *  - \p key_config->kid != NULL
 *  - \p key_config->common.perms != NULL
 *  - \p key_config->key_stor_opt is valid
 *  - \p key_config->kid_opt is valid
 *  - \p key_config->kid != \ref SDC_FLAG_INVALID_KID if \p key_config->kid_opt == \ref SDC_CREATE_KEY_FIXED_ID
 *  - \p key_config->common.fmt is a valid member of \ref sdc_key_fmt_t
 *  - \p key_config->common.len is a valid member of \ref sdc_key_len_t and supported for the given \p key_config->common.fmt of the architecture
 *  - \p key_secret->enc is a valid member of \ref sdc_keysecret_enc_t
 *  - \p key_secret->secret != NULL
 *  - \p key_secret->secret_len > 0
 */
SDC_ARCH_API sdc_error_t sdc_arch_insert_storage_key(
    const sdc_keystore_key_cfg_t *key_config,
    const sdc_keysecret_t *key_secret);

/**
 * \ingroup arch_keystore
 * \brief Architecture specific function to generate a random, symmetric keystore key.
 *
 * Note: The system needs to guarantee that all persistent keys of the previous power cycle are restored
 * before any storage key can be installed or removed. While the keystore is being restored this function
 * needs to return \ref SDC_KEY_NOT_READY.
 *
 * Note: The owner of a keystore key needs to be the current effective uid
 *       The architecture specific implementation needs to enforce
 *       uid in permissions equals current effective uid
 *
 * The implementation of this function needs to return only after the key has been created and
 * in case of a persistent key stored to disk.
 * In case of a reboot during key installation it has to be guaranteed for persistent key that
 *  - the key is installed correctly (with correct settings, data etc.)
 *  - not available at all
 * Furthermore the installation of the key needs to guarantee internally that any (i.e. parallel running)
 * process can't reference or use the key (e.g. \ref sdc_session_load_storage_key) before the
 * key has created completely and in case of a persistent key been stored to disk.
 * This is required to guarantee, that even in case of reboot during key installation no data can
 * e.g. encrypted with a key which is not available after reboot.
 *
 * Note: the uid of the key is the effective uid of the requesting process.
 * Note: the gid of the key can be specified using \p permissions. Any gid can be used. Even
 * if the gid is neither the effective gid of the process nor in its supplementary groups.
 *
 * Note: in case the key ID is determined automatically \p kid_opt == \ref SDC_CREATE_KEY_AUTOMATIC_ID
 * the architecture dependent function needs to return a free id in the range of
 * \ref SDC_AUTOMATIC_KID_RANGE_MIN and \ref SDC_AUTOMATIC_KID_RANGE_MAX
 *
 * Please refer to \ref sdc_generate_random_storage_key for details on the parameters and return values.
 * The architecture independent wrapper will assure that
 *  - \p key_config->kid != NULL
 *  - \p key_config->common.perms != NULL
 *  - \p key_config->key_stor_opt is valid
 *  - \p key_config->kid_opt is valid
 *  - \p key_config->kid != \ref SDC_FLAG_INVALID_KID if \p key_config.kid_opt == \ref SDC_CREATE_KEY_FIXED_ID
 *  - \p key_config->common.fmt is a valid member of \ref sdc_key_fmt_t
 *  - \p key_config->common.len is a valid member of \ref sdc_key_len_t and supported for the given \p key_config->common.fmt of the architecture
 */
SDC_ARCH_API sdc_error_t sdc_arch_generate_sym_storage_key(
    const sdc_keystore_key_cfg_t *key_config);

/**
 * \ingroup arch_keystore
 * \brief Architecture specific function to import a key to key_storage..
 *
 * Note: The system needs to guarantee that all persistent keys of the previous power cycle are restored
 * before any storage key can be installed or removed. While the keystore is being restored this function
 * needs to return \ref SDC_KEY_NOT_READY.
 *
 * Note: The owner of a keystore key needs to be the current effective uid
 *       The architecture specific implementation needs to enforce
 *       uid in permissions equals current effective uid
 *
 * The implementation of this function needs to return only after the key has been created and
 * in case of a persistent key stored to disk.
 * In case of a reboot during key installation it has to be guaranteed for persistent key that
 *  - the key is installed correctly (with correct settings, data etc.)
 *  - not available at all
 * Furthermore the installation of the key needs to guarantee internally that any (i.e. parallel running)
 * process can't reference or use the key (e.g. \ref sdc_session_load_storage_key) before the
 * key has created completely and in case of a persistent key been stored to disk.
 * This is required to guarantee, that even in case of reboot during key installation no data can
 * e.g. encrypted with a key which is not available after reboot.
 *
 * Note: the uid of the key is the effective uid of the requesting process.
 * Note: the gid of the key can be specified using \p permissions. Any gid can be used. Even
 * if the gid is neither the effective gid of the process nor in its supplementary groups.
 *
 * Note: in case the key ID is determined automatically \p kid_opt == \ref SDC_CREATE_KEY_AUTOMATIC_ID
 * the architecture dependent function needs to return a free id in the range of
 * \ref SDC_AUTOMATIC_KID_RANGE_MIN and \ref SDC_AUTOMATIC_KID_RANGE_MAX
 *
 * NOTE: No element of sdc_permissions_t must exceed the upper limit
 * specified by the inheritance mask of the key(currently loaded) used to wrap \p wrapped_key.
 *
 * Please refer to \ref sdc_import_wrapped_storage_key for details on the parameters and return values.
 * The architecture independent wrapper will assure that
 *  - \p session != NULL
 *  - \p key_config->kid != NULL
 *  - \p key_config->common.perms != NULL
 *  - \p key_config->key_stor_opt is valid
 *  - \p key_config->kid_opt is valid
 *  - \p key_config->kid != \ref SDC_FLAG_INVALID_KID if \p key_config.kid_opt == \ref SDC_CREATE_KEY_FIXED_ID
 *  - \p key_config->common.fmt is a valid member of \ref sdc_key_fmt_t
 *  - \p key_config->common.len is a valid member of \ref sdc_key_len_t and supported for the given \p key_config->common.fmt of the architecture
 *  - \p wrapped_key->enc is a valid member of \ref sdc_keysecret_enc_t
 *  - \p wrapped_key->iv != NULL if \p wraped_key->iv_len != 0
 *  - \p wrapped_key->wrapped_data != NULL if \p wrapped_key->wrapped_len != 0
 *  - \p wrapped_key->wrap_desc is the descriptor for wrapped_key->wrap_type
 */
SDC_ARCH_API sdc_error_t sdc_arch_import_wrapped_storage_key(
    sdc_session_t *session,
    const sdc_keystore_key_cfg_t *key_config,
    const sdc_wrapped_key_t *wrapped_key);

/**
 * \ingroup arch_keystore
 * \brief Architecture specific function to remove a keystore key.
 *
 * Note: The system needs to guarantee that all persistent keys of the previous power cycle are restored
 * before any storage key can be installed or removed. While the keystore is being restored this function
 * needs to return \ref SDC_KEY_NOT_READY.
 *
 * The implementation of this function needs to return only after the key has been deleted and
 * in case of a persistent key removed from disk.
 * In case of a reboot during key removal it has to be guaranteed for persistent key that
 *  - the key is deleted completely
 *  - not modified at all
 * Furthermore the removal of the key needs to guarantee internally that any (i.e. parallel running)
 * process can't reference or use the key (e.g. \ref sdc_session_load_storage_key) while the
 * removal is ongoing and that removal is only started when the key is not assigned
 * to any session (incl. sessions of other processes).
 *
 * Note: In order to remove a storage key, the requesting process needs to have remove permission.
 * This permission might be granted by the effective uid of the process (matching the uid of the key) or
 * by any gid (effective gid of process or any supplementary group) (matching the gid of the key).
 * Until permissions are modeled by flags (see \ref sdc_permissions) only matching uids are able to remove a key.
 *
 * Please refer to \ref sdc_remove_storage_key for details on the parameters and return values.
 * The architecture independent wrapper will assure that \p kid != \ref SDC_FLAG_INVALID_KID
 */
SDC_ARCH_API sdc_error_t sdc_arch_remove_storage_key(sdc_key_id_t kid);


/**
 * \ingroup arch_session
 * \brief Architecture specific function to initialize architecture specific session content.
 *
 * This function will be called during \ref sdc_open_session to initialize
 * the architecture specific session content (e.g. open a file descriptor)
 * Please refer to \ref sdc_open_session for details on the parameters and return values.
 * The architecture independent wrapper will assure that \p arch_session != NULL.
 *
 * Note: In case this function returns an error \ref sdc_arch_session_deinit won't be called.
 * All potentially internally allocated memory needs to be freed before.
 */
SDC_ARCH_API sdc_error_t sdc_arch_session_init(sdc_session_t *session);

/**
 * \ingroup arch_session
 * \brief Architecture specific function to deinitialize architecture specific session content.
 *
 * This function will be called during \ref sdc_close_session to deinitialize
 * the architecture specific session content (e.g. close file descriptor,
 * free internally allocated memory)
 * Please refer to \ref sdc_close_session for details on the parameters and return values.
 * The architecture independent wrapper will assure that \p arch_session != NULL.
 */
SDC_ARCH_API sdc_error_t sdc_arch_session_deinit(sdc_session_t *session);

/**
 * \ingroup arch_session
 * \brief Architecture specific function to assign the key storage key with the given ID to the session.
 *
 * Note: The system needs to guarantee that all persistent keys of the previous power cycle are restored
 * before any storage key can be assigned to any session. While the keystore is being restored this function
 * needs to return \ref SDC_KEY_NOT_READY.
 * When trying load a key which is currently being installed the system might return
 * \ref SDC_KEY_NOT_READY or \ref SDC_KID_NOT_AVAILABLE.
 * Please refer to \ref sdc_arch_insert_storage_key, \ref sdc_arch_generate_sym_storage_key and
 * \ref sdc_arch_remove_storage_key.
 *
 * Note: In order to load a storage key, the requesting process needs to be able to perform at least one
 * operation. This permission might be granted by the effective uid of the process (matching the uid of the key) or
 * by any gid (effective gid of process or any supplementary group) (matching the gid of the key).
 *
 * Please refer to \ref sdc_session_load_storage_key for details on the parameters and return values.
 * The architecture independent wrapper will assure that \p session != NULL
 * and \p kid != \ref SDC_FLAG_INVALID_KID.
 */
SDC_ARCH_API sdc_error_t sdc_arch_session_load_storage_key(sdc_session_t *session, sdc_key_id_t kid);

/* Note: validity of session needs to be checked internally */
/**
 * \ingroup arch_session
 * \brief Architecture specific function to prepare a built in key for usage.
 *
 * This function is used to generate and assign a built in key to the session.
 * All cryptographic operations (see \ref operations) will use this key until the
 * key is unset or replaced by a different key.
 *
 * \param[in] session          session handle created using ::sdc_open_session
 * \param[in] key_config       format, length and permissions of the key
 * \param[in] key_modifier         modifier data
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ACCESS_DENIED - the process has no permission to perform any operation using the given key
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_ARCH_API sdc_error_t sdc_arch_session_load_builtin_key(sdc_session_t *session,
                                                           const sdc_key_common_t *key_config,
                                                           const sdc_key_modifier_t *key_modifier);


/**
 * \ingroup arch_session
 * \brief Architecture specific function to remove assigned key from session.
 *
 * Please refer to \ref sdc_session_unset_key for details on the parameters and return values.
 * The architecture independent wrapper will assure that \p session != NULL.
 */
SDC_ARCH_API sdc_error_t sdc_arch_session_unset_key(sdc_session_t *session);

/**
 * \ingroup arch_session
 * \brief Architecture specific function to fill the key info
 *
 * \param[in] session          session handle created using ::sdc_open_session
 * \param[in] key_info         info struct to fill - already reset
 * \param[in] with_modifier    allocate modifier memory and read modifier
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_KEY_UNSET - in case no session key is set
 * \return \ref SDC_KEY_INFO_INVALID - the key info returned by e.g. the kernel is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory for modifier
 */
SDC_ARCH_API sdc_error_t sdc_arch_session_key_info_get(sdc_session_t *session,
                                                       sdc_key_info_t *key_info, bool with_modifier);


/**
 * \ingroup arch_random
 * \brief Architecture specific random number generation
 *
 * This function takes a pointer to a preallocated buffer and fills it with the
 * specified amount of random data.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_random_gen(sdc_session_t *session,
                                             const size_t len,
                                             uint8_t *random_data);

/**
 * \ingroup arch_wrapunwrap
 * \defgroup arch_wrapunwrap_type Type
 * Type related functions,types etc.
 */

/**
 * \struct sdc_wrap_unwrap_type
 * \ingroup arch_wrapunwrap_type
 * \brief Architecture specific struct to store the currently selected type for wrap/unwrap.
 *
 * This type needs to store which \ref sdc_wrap_unwrap_alg_t and \ref sdc_wrap_unwrap_blk_t
 * has been selected by the user.
 * Furthermore it might be used to store the architecture specific representation of these
 * values or any other information related to the selected type.
 */

/**
 * \ingroup arch_wrapunwrap_type
 * \brief Return the default type for wrap/unwrap in this architecture
 *
 * Please refer to \ref sdc_wrap_unwrap_get_default for details.
 */
SDC_ARCH_API const sdc_wrap_unwrap_type_t *sdc_arch_wrap_unwrap_get_default(void);

/**
 * \ingroup arch_wrapunwrap_type
 * \brief Architecture specific allocation of \ref sdc_wrap_unwrap_type
 *
 * Please refer to \ref sdc_wrap_unwrap_type_alloc for details.
 * The architecture independent wrapper will assure that \p type != NULL.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_wrap_unwrap_type_alloc(sdc_wrap_unwrap_type_t **type);

/**
 * \ingroup arch_wrapunwrap_type
 * \brief Architecture specific deallocation of \ref sdc_wrap_unwrap_type
 *
 * Please refer to \ref sdc_wrap_unwrap_type_free for details.
 * The architecture independent wrapper will assure that \p type != NULL.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_wrap_unwrap_type_free(sdc_wrap_unwrap_type_t *type);

/**
 * \ingroup arch_wrapunwrap_type
 * \brief Architecture specific function to set \ref sdc_wrap_unwrap_alg_t
 *
 * Please refer to \ref sdc_wrap_unwrap_type_set_algorithm for details.
 * The architecture independent wrapper will assure that \p type != NULL and
 * \p alg in the range of \ref SDC_WRAP_ALG_FIRST and \ref SDC_WRAP_ALG_END - 1.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_wrap_unwrap_type_set_algorithm(sdc_wrap_unwrap_type_t *type, sdc_wrap_unwrap_alg_t alg);

/**
 * \ingroup arch_wrapunwrap_type
 * \brief Architecture specific function to set \ref sdc_wrap_unwrap_blk_t
 *
 * Please refer to \ref sdc_wrap_unwrap_type_set_block_mode for details.
 * The architecture independent wrapper will assure that \p type != NULL and
 * \p blk in the range of \ref SDC_WRAP_BLK_FIRST and \ref SDC_WRAP_BLK_END - 1.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_wrap_unwrap_type_set_block_mode(sdc_wrap_unwrap_type_t *type, sdc_wrap_unwrap_blk_t blk);

/**
 * \ingroup arch_wrapunwrap_type
 * \brief Architecture specific function to read \ref sdc_wrap_unwrap_alg_t from \ref sdc_wrap_unwrap_type
 *
 * Please refer to \ref sdc_wrap_unwrap_type_get_algorithm for details.
 * The architecture independent wrapper will assure that \p type != NULL and \p alg != NULL.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_wrap_unwrap_type_get_algorithm(const sdc_wrap_unwrap_type_t *type, sdc_wrap_unwrap_alg_t *alg);

/**
 * \ingroup arch_wrapunwrap_type
 * \brief Architecture specific function to read \ref sdc_wrap_unwrap_blk_t from \ref sdc_wrap_unwrap_type
 *
 * Please refer to \ref sdc_wrap_unwrap_type_get_block_mode for details.
 * The architecture independent wrapper will assure that \p type != NULL and \p blk != NULL.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_wrap_unwrap_type_get_block_mode(const sdc_wrap_unwrap_type_t *type, sdc_wrap_unwrap_blk_t *blk);

/**
 * \ingroup arch_wrapunwrap_type
 * \brief Architecture specific function to fill \ref sdc_key_desc_t according to
 * the specified \ref sdc_wrap_unwrap_type_t.
 *
 * This function needs to be architecture specific as not every architecture might support all potential
 * parameter ranges as defined by the format(specified by \ref sdc_wrap_unwrap_type_t).
 *
 * The architecture independent wrapper will assure that \p type != NULL and \p key_desc != NULL.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_wrap_unwrap_key_desc_fill (
    const sdc_wrap_unwrap_type_t *type,
    sdc_key_desc_t *key_desc);

/**
 * \ingroup arch_wrapunwrap_type
 * \brief Architecture specific function to fill \ref sdc_wrap_unwrap_desc_t according to
 * the specified \ref sdc_wrap_unwrap_type_t and data length
 *
 * This function needs to be architecture specific as not every architecture might support all potential
 * parameter ranges as defined by the format(specified by \ref sdc_wrap_unwrap_type_t).
 *
 * The architecture independent wrapper will assure that \p type != NULL and \p desc != NULL.
 *
 * Note: session might be NULL.
 * In case the session is required to fill the descriptor (e.g. tag length depends on
 * loaded key length) the function needs to return SDC_SESSION_INVALID;
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_wrap_unwrap_desc_fill (
    sdc_session_t *session,
    const sdc_wrap_unwrap_type_t *type,
    sdc_wrap_unwrap_desc_t *desc,
    size_t total_data_len);

/**
 * \ingroup arch_wrapunwrap
 * \defgroup arch_wrapunwrap_op Operations
 * Operation related functions etc.
 */

/**
 * \ingroup arch_wrapunwrap_op
 * \defgroup arch_wrapunwrap_op_iuf Init,Update,Finalize
 * Perform operation for the currently provided part using Init, Update, Finalize
 */

/**
 * \ingroup arch_wrapunwrap_op_iuf
 * \defgroup arch_wrap_op_iuf Init,Update,Finalize (wrap)
 * Perform wrap operation using Init, Update, Finalize
 */

/**
 * \ingroup arch_wrapunwrap_op_iuf
 * \defgroup arch_unwrap_op_iuf Init,Update,Finalize ((unwrap)
 * Perform unwrap operation using Init, Update, Finalize
 */

/**
 * \ingroup arch_wrapunwrap_op
 * \defgroup arch_wrapunwrap_op_aio All in one
 * Perform operation on the complete content in a single shot
 */

/**
 * \ingroup arch_wrap_op_iuf
 * \brief Architecture specific function to initialize init, update, finalize
 * wrap sequence
 *
 * This function maps very closely to \ref sdc_wrap_init as available to applications
 * \p iv, \p iv_len have either been provided by application or automatically generated
 * in the common part.\n
 * \p session holds wrap specific parameters like total input length, total aad length  and tag length.
 * \n
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code already checked that \p iv, \p iv_len are valid for the given \p type.\n
 * \n
 * Please refer to \ref sdc_wrap_init for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_wrap_init
 * \param[in] type same as \p type of \ref sdc_wrap_init
 * \param[in] desc - filled \ref sdc_wrap_unwrap_desc_t descriptor
 * \param[in] iv       pointer to the IV buffer
 * \param[in] iv_len   length of the IV\n
 *
 * Additional error codes
 * \return refer to \ref sdc_wrap_init
 */
SDC_ARCH_API sdc_error_t sdc_arch_wrap_init(sdc_session_t *session,
                                            const sdc_wrap_unwrap_type_t *type,
                                            const sdc_wrap_unwrap_desc_t *desc,
                                            const uint8_t *iv, size_t iv_len);

/**
 * \ingroup arch_unwrap_op_iuf
 * \brief Architecture specific function to initialize init, update, finalize
 * unwrap sequence
 *
 * This function maps very closely to \ref sdc_unwrap_init as available to applications\n
 * \p session holds unwrap specific parameters like total input length, total aad length  and tag length.\n
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code already checked that \p iv, \p iv_len are valid for the given \p type.\n
 * \n
 * Please refer to \ref sdc_unwrap_init for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_unwrap_init
 * \param[in] type same as \p type of \ref sdc_unwrap_init
 * \param[in] desc - filled \ref sdc_wrap_unwrap_desc_t descriptor
 * \param[in] iv same as \p iv of \ref sdc_unwrap_init
 * \param[in] iv_len length of \p iv
 *
 * Additional error codes
 * \return refer to \ref sdc_unwrap_init
 * \return SDC_NOT_SUPPORTED - non-supported algorithm/mode
 */
SDC_ARCH_API sdc_error_t sdc_arch_unwrap_init(sdc_session_t *session,
                                              const sdc_wrap_unwrap_type_t *type,
                                              const sdc_wrap_unwrap_desc_t *desc,
                                              const uint8_t *iv, size_t iv_len);

/**
 * \ingroup arch_wrap_op_iuf
 * \brief Architecture specific function to wrap data using init, update, finalize
 *
 * This function maps very closely to \ref sdc_wrap_aad_update as available to applications.
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code already checked that \p aad_data is not NULL.\n
 * The common code will guarantee that this function is only called with
 * \p aad_len >= 0.\n
 * Please refer to \ref sdc_wrap_aad_update for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_wrap_aad_update
 * \param[in] type same as \p type of \ref sdc_wrap_aad_update
 * \param[in] desc - filled \ref sdc_wrap_unwrap_desc_t descriptor
 * \param[in] aad_data same as \p in_data of \ref sdc_wrap_aad_update
 * \param[in] aad_len same as \p in_len of \ref sdc_wrap_aad_update
 *
 * Additional error codes
 * \return refer to \ref sdc_wrap_aad_update
 */
SDC_ARCH_API sdc_error_t sdc_arch_wrap_aad_update(sdc_session_t *session,
                                                  const sdc_wrap_unwrap_type_t *type,
                                                  const sdc_wrap_unwrap_desc_t *desc,
                                                  const uint8_t *aad_data, const size_t aad_len);

/**
 * \ingroup arch_unwrap_op_iuf
 * \brief Architecture specific function to unwrap data using init, update, finalize
 *
 * This function maps very closely to \ref sdc_unwrap_aad_update as available to applications.
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code already checked that \p aad_data is not NULL.\n
 * The common code will guarantee that this function is only called with
 * \p aad_len >= 0.\n
 * Please refer to \ref sdc_unwrap_aad_update for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_unwrap_aad_update
 * \param[in] type same as \p type of \ref sdc_unwrap_aad_update
 * \param[in] desc - filled \ref sdc_wrap_unwrap_desc_t descriptor
 * \param[in] aad_data same as \p in_data of \ref sdc_unwrap_aad_update
 * \param[in] aad_len same as \p in_len of \ref sdc_unwrap_aad_update
 *
 * Additional error codes
 * \return refer to \ref sdc_wrap_aad_update
 */
SDC_ARCH_API sdc_error_t sdc_arch_unwrap_aad_update(sdc_session_t *session,
                                                    const sdc_wrap_unwrap_type_t *type,
                                                    const sdc_wrap_unwrap_desc_t *desc,
                                                    const uint8_t *aad_data, const size_t aad_len);

/**
 * \ingroup arch_wrap_op_iuf
 * \brief Architecture specific function to wrap data using init, update, finalize
 *
 * This function maps very closely to \ref sdc_wrap_update as available to applications.
 * The function needs to internally verify that the output buffer provided by
 * application needs to be sufficient large to handle the provided input.
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code already checked that \p in_data is not NULL
 * Note: The common code already checked that \p out_len and \p out_data_len are not NULL.\n
 * \n
 * The common code will guarantee that this function is only called with
 * 0 < \p in_data_len <= \p desc->data.max_chunk_len.\n
 * In case \p desc->data.chunk_len_aligned is true in_data_len is furthermore
 * aligned to desc->data.block_len.\n
 * In this case handling unaligned data length is handled in the common code.
 * Therefore a single \ref sdc_wrap_update call by application can result in
 * multiple \ref sdc_arch_wrap_update calls.
 * \n
 * Please refer to \ref sdc_wrap_update for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_wrap_update
 * \param[in] type same as \p type of \ref sdc_wrap_update
 * \param[in] desc - filled \ref sdc_wrap_unwrap_desc_t descriptor
 * \param[in] in_data slice of input of \ref sdc_wrap_update
 * \param[in] in_data_len slice length of \p in_data
 * \param[out] out_data pointer to the output buffer.
 * \param[in,out] out_data_len  pointer to the output buffer length\n
 *                              input: available length / output: used length\n
 *
 * Additional error codes
 * \return refer to \ref sdc_wrap_update
 */
SDC_ARCH_API sdc_error_t sdc_arch_wrap_update(sdc_session_t *session,
                                              const sdc_wrap_unwrap_type_t *type,
                                              const sdc_wrap_unwrap_desc_t *desc,
                                              const uint8_t *in_data, const size_t in_data_len,
                                              uint8_t *out_data, size_t *out_data_len);

/**
 * \ingroup arch_unwrap_op_iuf
 * \brief Architecture specific function to unwrap data using init, update, finalize
 *
 * This function maps very closely to \ref sdc_unwrap_update as available to applications.
 * The function needs to internally verify that the output buffer provided by
 * application needs to be sufficient large to handle the provided input.
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code already checked that \p in_data is not NULL
 * Note: The common code already checked that \p out_len and \p out_data_len are not NULL.\n
 * \n
 * The common code will guarantee that this function is only called with
 * 0 < \p in_data_len <= \p desc->data.max_chunk_len.\n
 * In case \p desc->data.chunk_len_aligned is true in_data_len is furthermore
 * aligned to desc->data.block_len.\n
 * In this case handling unaligned data length is handled in the common code.
 * Therefore a single \ref sdc_unwrap_update call by application can result in
 * multiple \ref sdc_arch_unwrap_update calls.
 * \n
 * Please refer to \ref sdc_unwrap_update for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_unwrap_update
 * \param[in] type same as \p type of \ref sdc_unwrap_update
 * \param[in] desc - filled \ref sdc_wrap_unwrap_desc_t descriptor
 * \param[in] in_data slice of input of \ref sdc_unwrap_update
 * \param[in] in_data_len slice length of \p in_data
 * \param[out] out_data pointer to the output buffer.
 * \param[in,out] out_data_len  pointer to the output buffer length\n
 *                              input: available length / output: used length\n
 *
 * Additional error codes
 * \return refer to \ref sdc_unwrap_update
 */
SDC_ARCH_API sdc_error_t sdc_arch_unwrap_update(sdc_session_t *session,
                                                const sdc_wrap_unwrap_type_t *type,
                                                const sdc_wrap_unwrap_desc_t *desc,
                                                const uint8_t *in_data, const size_t in_data_len,
                                                uint8_t *out_data, size_t *out_data_len);

/**
 * \ingroup arch_wrap_op_iuf
 * \brief Architecture specific function to finalize an wrap init, update, finalize sequence
 *
 * This function maps very closely to \ref sdc_wrap_finalize as available to applications.
 * The function needs to internally verify that the output buffer provided by
 * application needs to be sufficient large to handle the provided input.
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code assures that \p in_data is not NULL if \p in_data_len != 0
 * Note: The common code already checked that \p out_len and \p out_data_len are not NULL.\n
 * \n
 * During update the common code will guarantee that \ref sdc_wrap_update is only
 * called with 0 < \p in_data_len <= \p desc->data.max_chunk_len and
 * \p in_data_len fitting to \p desc->data.chunk_len_aligned.\n
 * The potential remaining unaligned part is passed by common code when
 * calling \ref sdc_arch_wrap_finalize.\n
 * \n
 * Please refer to \ref sdc_wrap_finalize for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_wrap_finalize
 * \param[in] type same as \p type of \ref sdc_wrap_finalize
 * \param[in] desc - filled \ref sdc_wrap_unwrap_desc_t descriptor
 * \param[in] in_data potential unaligned data not handled by previous \ref sdc_arch_wrap_update
 * \param[in] in_data_len length of \p in_data - 0 <= \p in_data <= desc->data.block_len
 * \param[out] tag_data pointer to the tag buffer.
 * \param[in] tag_len length of tag buffer.\n
 * \param[out] out_data pointer to the output buffer.
 * \param[in,out] out_data_len  pointer to the output buffer length\n
 *                              input: available length / output: used length\n
 *
 * Additional error codes
 * \return refer to \ref sdc_wrap_finalize
 */
SDC_ARCH_API sdc_error_t sdc_arch_wrap_finalize(sdc_session_t *session,
                                                const sdc_wrap_unwrap_type_t *type,
                                                const sdc_wrap_unwrap_desc_t *desc,
                                                const uint8_t *in_data, const size_t in_data_len,
                                                uint8_t *tag_data, size_t tag_len,
                                                uint8_t *out_data, size_t *out_data_len);

/**
 * \ingroup arch_unwrap_op_iuf
 * \brief Architecture specific function to finalize an unwrap init, update, finalize sequence
 *
 * This function maps very closely to \ref sdc_unwrap_finalize as available to applications.
 * The function needs to internally verify that the output buffer provided by
 * application needs to be sufficient large to handle the provided input.
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code assures that \p in_data is not NULL if \p in_data_len != 0
 * Note: The common code already checked that \p out_len and \p out_data_len are not NULL.\n
 * \n
 * During update the common code will guarantee that \ref sdc_unwrap_update is only
 * called with 0 < \p in_data_len <= \p desc->data.max_chunk_len and
 * \p in_data_len fitting to \p desc->data.chunk_len_aligned.\n
 * The potential remaining unaligned part is passed by common code when
 * calling \ref sdc_arch_unwrap_finalize.\n
 * \n
 * 0 <= \p in_data <= desc->data.block_len\n (0 if \p chunk_len_aligned == false)\n
 * Else \p in_data == desc->data.block_len\n*
 *
 * Please refer to \ref sdc_unwrap_finalize for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_unwrap_finalize
 * \param[in] type same as \p type of \ref sdc_unwrap_finalize
 * \param[in] desc - filled \ref sdc_wrap_unwrap_desc_t descriptor
 * \param[in] in_data potential unaligned data not handled by previous \ref sdc_arch_unwrap_update
 * \param[in] in_data_len length of \p in_data - 0 <= \p in_data <= desc->data.block_len
 * \param[in] tag_data pointer to the tag buffer.
 * \param[in] tag_len length of tag buffer.\n
 * \param[out] out_data pointer to the output buffer.
 * \param[in,out] out_data_len  pointer to the output buffer length\n
 *                              input: available length / output: used length\n
 *
 * Additional error codes
 * \return refer to \ref sdc_unwrap_finalize
 */
SDC_ARCH_API sdc_error_t sdc_arch_unwrap_finalize(sdc_session_t *session,
                                                  const sdc_wrap_unwrap_type_t *type,
                                                  const sdc_wrap_unwrap_desc_t *desc,
                                                  const uint8_t *in_data, const size_t in_data_len,
                                                  const uint8_t *tag_data, size_t tag_len,
                                                  uint8_t *out_data, size_t *out_data_len);

/**
 * \ingroup arch_wrapunwrap_op_aio
 * \brief Architecture specific to calculate exact length for cipher output
 *
 *  For encrypt and wrap the common code needs to allocate memory to store the result.
 *  This data needs to be large enough to contain encrypted/wrapped plain data + padding
 *
 * \param[in] session same as \p session of \ref sdc_unwrap
 * \param[in] type same as \p type of \ref sdc_unwrap
 * \param[in] desc - filled \ref sdc_wrap_unwrap_desc_t descriptor
 * \param[in] in_len length of the plain data to wrap
 * \param[out] out_len upper length of cipher output (wrapped plain data + padding)
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INTERNAL_ERROR - this probably means that a new value has been added to sdc_padding_t without handling it here
 */
SDC_ARCH_API sdc_error_t sdc_arch_wrap_get_out_len(sdc_session_t *session,
                                                   const sdc_wrap_unwrap_type_t *type,
                                                   const sdc_wrap_unwrap_desc_t *desc,
                                                   const size_t in_len,
                                                   size_t *out_len);

/**
 * \ingroup arch_wrapunwrap_op_aio
 * \brief Architecture specific function to execute an optimized
 * wrapping operation
 *
 * Architecture specific function to execute an optimized
 * wrapping operation instead of using \ref sdc_wrap_init,
 * \ref sdc_wrap_update, \ref sdc_wrap_finalize.
 * In case the architecture does not provide the optimization this function
 * has to return \ref SDC_NOT_SUPPORTED
 * In this case the common part of SDC library will use the mandatory functions
 * \ref sdc_wrap_init, \ref sdc_wrap_update, \ref sdc_wrap_finalize
 * for realization
 *
 * This function maps very closely to \ref sdc_wrap as available to applications
 * In contrast to this function \p desc is provided and the type of \p out_data and
 * \p tag_out_data are simple pointers (memory is allocated in surrounding common wrapper).
 * This memory is sufficient large to even include padding if required.
 *
 * Last but not least most NULL pointer checks have already been performed.
 *
 * Note: Adding padding needs to be done inside to allow optimization
 *
 * Please refer to \ref sdc_wrap for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_wrap
 * \param[in] type same as \p type of \ref sdc_wrap
 * \param[in] desc - filled \ref sdc_wrap_unwrap_desc_t descriptor
 * \param[in] in_data same as \p in_data of \ref sdc_wrap
 * \param[in] in_data_len same as \p in_len of \ref sdc_wrap
 * \param[out] out_data pointer to the output buffer (allocated by \ref sdc_wrap)
 * \param[in] out_data_len length of the output buffer (allocated by \ref sdc_wrap) - aligned to blocklength if required (see \p desc)
 * \param[in] aad_data same as \p aad_data of \ref sdc_wrap
 * \param[in] aad_len same as \p aad_len of \ref sdc_wrap
 * \param[in] iv either the \p iv passed to \ref sdc_wrap or the internally generated iv
 * \param[in] iv_len length of \p iv
 * \param[out] tag_out_data pointer to the tag (mac or signature) buffer (allocated by \ref sdc_wrap)
 * \param[in] tag_out_len length of \p tag_out
 *
 * Additional error codes
 * \return refer to \ref sdc_wrap
 * \return SDC_NOT_SUPPORTED - no optimization implemented
 */
SDC_ARCH_API sdc_error_t sdc_arch_wrap(sdc_session_t *session,
                                       const sdc_wrap_unwrap_type_t *type,
                                       const sdc_wrap_unwrap_desc_t *desc,
                                       const uint8_t *in_data,
                                       const size_t in_data_len,
                                       uint8_t *out_data,
                                       const size_t out_data_len,
                                       const uint8_t *aad_data, const size_t aad_len,
                                       const uint8_t *iv, const size_t iv_len,
                                       uint8_t *tag_out_data, const size_t tag_out_len);

/**
 * \ingroup arch_wrapunwrap_op_aio
 * \brief Architecture specific to calculate upper limit for plain output
 *
 *  For decrypt and unwrap the common code needs to allocate memory to store the result.
 *  When using userspace padding and \ref sdc_intern_unpad this data needs to be large
 *  enough to hold data + padding.
 *
 * \param[in] session same as \p session of \ref sdc_unwrap
 * \param[in] type same as \p type of \ref sdc_unwrap
 * \param[in] desc - filled \ref sdc_wrap_unwrap_desc_t descriptor
 * \param[in] in_len length of the cipher data to unwrap
 * \param[out] out_max_len upper limit of plain output
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INTERNAL_ERROR - this probably means that a new value has been added to sdc_padding_t without handling it here
 */
SDC_ARCH_API sdc_error_t sdc_arch_unwrap_get_max_out_len(sdc_session_t *session,
                                                         const sdc_wrap_unwrap_type_t *type,
                                                         const sdc_wrap_unwrap_desc_t *desc,
                                                         const size_t in_len,
                                                         size_t *out_max_len);

/**
 * \ingroup arch_wrapunwrap_op_aio
 * \brief Architecture specific function to execute an optimized
 * unwrapping operation
 *
 * Architecture specific function to execute an optimized
 * unwrapping operation instead of using \ref sdc_unwrap_init,
 * \ref sdc_unwrap_update, \ref sdc_unwrap_finalize.
 * In case the architecture does not provide the optimization this function
 * has to return \ref SDC_NOT_SUPPORTED
 * In this case the common part of SDC library will use the mandatory functions
 * \ref sdc_unwrap_init, \ref sdc_unwrap_update, \ref sdc_unwrap_finalize
 * for realization
 *
 * This function maps very closely to \ref sdc_unwrap as available to applications
 * In contrast to this function \p desc is provided, input and output buffer
 * have a common length field (\p data_len) and the type of \p out_data is a
 * simple pointer (memory is allocated in surrounding common wrapper).
 * Last but not least most NULL pointer checks have already been performed.
 *
 * Note: In case the decryption type needs padding to the block alignment the
 * input data and the output data length is aligned to the block alignment (and include padding)
 * Removing the bytes padded during encrypt is handled in the surrounding function.
 *
 * Please refer to \ref sdc_unwrap for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_unwrap
 * \param[in] type same as \p type of \ref sdc_unwrap
 * \param[in] desc - filled \ref sdc_wrap_unwrap_desc_t descriptor
 * \param[in] in_data same as \p in_data of \ref sdc_wrap
 * \param[in] in_data_len length of \p in_data
 * \param[out] out_data pointer to the output buffer (allocated by \ref sdc_unwrap)
 * \param[in,out] out_data_len length of \p out_data - input avail buffer - output used buffer
 * \param[in] aad_data same as \p aad_data of \ref sdc_unwrap
 * \param[in] aad_len same as \p aad_len of \ref sdc_unwrap
 * \param[in] tag_in_data (mac or signature) same as \p tag_data \ref sdc_unwrap
 * \param[in] tag_in_len same as \p tag_len of \ref sdc_unwrap
 * \param[in] iv same as \p iv of \ref sdc_unwrap
 * \param[in] iv_len same as \p iv_len of \ref sdc_unwrap
 *
 * Additional error codes
 * \return refer to \ref sdc_unwrap
 * \return SDC_NOT_SUPPORTED - no optimization implemented
 */
SDC_ARCH_API sdc_error_t sdc_arch_unwrap(sdc_session_t *session,
                                         const sdc_wrap_unwrap_type_t *type,
                                         const sdc_wrap_unwrap_desc_t *desc,
                                         const uint8_t *in_data,
                                         const size_t in_data_len,
                                         uint8_t *out_data,
                                         size_t *out_data_len,
                                         const uint8_t *aad_data, const size_t aad_len,
                                         const uint8_t *tag_in_data, const size_t tag_in_len,
                                         const uint8_t *iv, const size_t iv_len);

/**
 * \ingroup arch_encdec
 * \defgroup arch_encdec_type Type
 * Type related functions,types etc.
 */

/**
 * \struct sdc_encrypt_decrypt_type
 * \ingroup arch_encdec_type
 * \brief Architecture specific struct to store the currently selected type for encrypt/decrypt.
 *
 * This type needs to store which \ref sdc_encrypt_decrypt_alg_t and \ref sdc_encrypt_decrypt_blk_t
 * has been selected by the user.
 * Furthermore it might be used to store the architecture specific representation of these
 * values or any other information related to the selected type.
 */

/**
 * \ingroup arch_encdec_type
 * \brief Return the default type for encrypt/decrypt in this architecture
 *
 * Please refer to \ref sdc_encrypt_decrypt_get_default for details.
 */
SDC_ARCH_API const sdc_encrypt_decrypt_type_t *sdc_arch_encrypt_decrypt_get_default(void);

/**
 * \ingroup arch_encdec_type
 * \brief Architecture specific allocation of \ref sdc_encrypt_decrypt_type_t
 *
 * Please refer to \ref sdc_encrypt_decrypt_type_alloc for details.
 * The architecture independent wrapper will assure that \p type != NULL.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_encrypt_decrypt_type_alloc(sdc_encrypt_decrypt_type_t **type);

/**
 * \ingroup arch_encdec_type
 * \brief Architecture specific deallocation of \ref sdc_encrypt_decrypt_type_t
 *
 * Please refer to \ref sdc_encrypt_decrypt_type_free for details.
 * The architecture independent wrapper will assure that \p type != NULL.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_encrypt_decrypt_type_free(sdc_encrypt_decrypt_type_t *type);

/**
 * \ingroup arch_encdec_type
 * \brief Architecture specific function to set \ref sdc_encrypt_decrypt_alg_t
 *
 * Please refer to \ref sdc_encrypt_decrypt_type_set_algorithm for details.
 * The architecture independent wrapper will assure that \p type != NULL and
 * \p alg in the range of \ref SDC_ENCDEC_ALG_FIRST and \ref SDC_ENCDEC_ALG_END - 1.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_encrypt_decrypt_type_set_algorithm(sdc_encrypt_decrypt_type_t *type, sdc_encrypt_decrypt_alg_t alg);

/**
 * \ingroup arch_encdec_type
 * \brief Architecture specific function to set \ref sdc_encrypt_decrypt_blk_t
 *
 * Please refer to \ref sdc_encrypt_decrypt_type_set_block_mode for details.
 * The architecture independent wrapper will assure that \p type != NULL and
 * \p blk in the range of \ref SDC_ENCDEC_BLK_FIRST and \ref SDC_ENCDEC_BLK_END - 1.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_encrypt_decrypt_type_set_block_mode(sdc_encrypt_decrypt_type_t *type, sdc_encrypt_decrypt_blk_t blk);

/**
 * \ingroup arch_encdec_type
 * \brief Architecture specific function to read \ref sdc_encrypt_decrypt_alg_t from \ref sdc_encrypt_decrypt_type_t
 *
 * Please refer to \ref sdc_encrypt_decrypt_type_get_algorithm for details.
 * The architecture independent wrapper will assure that \p type != NULL and \p alg != NULL.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_encrypt_decrypt_type_get_algorithm(const sdc_encrypt_decrypt_type_t *type, sdc_encrypt_decrypt_alg_t *alg);

/**
 * \ingroup arch_encdec_type
 * \brief Architecture specific function to read \ref sdc_encrypt_decrypt_blk_t from \ref sdc_encrypt_decrypt_type_t
 *
 * Please refer to \ref sdc_encrypt_decrypt_type_get_block_mode for details.
 * The architecture independent wrapper will assure that \p type != NULL and \p blk != NULL.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_encrypt_decrypt_type_get_block_mode(const sdc_encrypt_decrypt_type_t *type, sdc_encrypt_decrypt_blk_t *blk);

/**
 * \ingroup arch_encdec_type
 * \brief Architecture specific function to fill \ref sdc_key_desc_t  according to
 * the specified \ref sdc_encrypt_decrypt_type_t.
 *
 * This function needs to be architecture specific as not every architecture might support all potential
 * parameter ranges as defined by the format(specified by \ref sdc_encrypt_decrypt_type_t).
 *
 * The architecture independent wrapper will assure that \p type != NULL and \p desc != NULL.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_encrypt_decrypt_key_desc_fill (
    const sdc_encrypt_decrypt_type_t *type,
    sdc_key_desc_t *desc);

/**
 * \ingroup arch_encdec_type
 * \brief Architecture specific function to fill \ref sdc_encrypt_decrypt_desc_t according to
 * the specified \ref sdc_encrypt_decrypt_type_t.
 *
 * This function needs to be architecture specific as not every architecture might support all potential
 * parameter ranges as defined by the format(specified by \ref sdc_encrypt_decrypt_type_t).
 *
 * The architecture independent wrapper will assure that \p type != NULL and \p desc != NULL.
 *
 * Note: session might be NULL.
 * In case the session is required to fill the descriptor (e.g. tag length depends on
 * loaded key length) the function needs to return SDC_SESSION_INVALID;
 */

SDC_ARCH_API sdc_error_t sdc_arch_encrypt_decrypt_desc_fill (
    sdc_session_t *session,
    const sdc_encrypt_decrypt_type_t *type,
    sdc_encrypt_decrypt_desc_t *desc);


/**
 * \ingroup arch_encdec
 * \defgroup arch_encdec_op Operations
 * Operation related functions etc.
 */

/**
 * \ingroup arch_encdec_op
 * \defgroup arch_encdec_op_iuf Init,Update,Finalize
 * Init, Update, Finalize API for encrypt/decrpyt
 */

/**
 * \ingroup arch_encdec_op_iuf
 * \defgroup arch_enc_op_iuf Init,Update,Finalize (Encrypt)
 * Perform encrypt operation using Init, Update, Finalize
 */

/**
 * \ingroup arch_encdec_op_iuf
 * \defgroup arch_dec_op_iuf Init,Update,Finalize ((Decrypt)
 * Perform decrypt operation using Init, Update, Finalize
 */

/**
 * \ingroup arch_encdec_op
 * \defgroup arch_encdec_op_aio All in one
 * Perform operation on the complete content in a single shot
 */

/**
 * \ingroup arch_encdec_op_aio
 * \brief Architecture specific to calculate exact length for cipher output
 *
 *  For encrypt and wrap the common code needs to allocate memory to store the result.
 *  This data needs to be large enough to contain encrypted/wrapped plain data + padding
 *
 * \param[in] session same as \p session of \ref sdc_encrypt
 * \param[in] type same as \p type of \ref sdc_encrypt
 * \param[in] desc - filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] in_len length of the plain data to encrypt
 * \param[out] out_len upper length of cipher output (encrypted plain data + padding)
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INTERNAL_ERROR - this probably means that a new value has been added to sdc_padding_t without handling it here
 */
SDC_ARCH_API sdc_error_t sdc_arch_encrypt_get_out_len(sdc_session_t *session,
                                         const sdc_encrypt_decrypt_type_t *type,
                                         const sdc_encrypt_decrypt_desc_t *desc,
                                         const size_t in_len,
                                         size_t *out_len);

/**
 * \ingroup arch_encdec_op_aio
 * \brief Architecture specific function to execute an optimized
 * encrypt operation
 *
 * Architecture specific function to execute an optimized
 * encrypt operation instead of using \ref sdc_encrypt_init,
 * \ref sdc_encrypt_update, \ref sdc_encrypt_finalize.
 * In case the architecture does not provide the optimization this function
 * has to return \ref SDC_NOT_SUPPORTED
 * In this case the common part of SDC library will use the mandatory functions
 * \ref sdc_encrypt_init, \ref sdc_encrypt_update, \ref sdc_encrypt_finalize
 * for realization
 *
 * This function maps very closely to \ref sdc_encrypt as available to applications
 * In contrast to this function \p desc is provided and the type of \p out_data is
 * a simple pointer (memory is allocated in surrounding common wrapper). This
 * memory is sufficient large to even include padding if required.
 *
 * Last but not least most NULL pointer checks have already been performed.
 *
 * Note: Adding padding needs to be done inside to allow optimization
 *
 * Please refer to \ref sdc_encrypt for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_encrypt
 * \param[in] type same as \p type of \ref sdc_encrypt
 * \param[in] desc - filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] in_data same as \p in_data of \ref sdc_encrypt
 * \param[in] in_data_len same as \p in_len of \ref sdc_encrypt
 * \param[out] out_data pointer to the output buffer (allocated by \ref sdc_encrypt)
 * \param[in] out_data_len length of the output buffer (allocated by \ref sdc_encrypt) - aligned to blocklength if required (see \p desc)
 * \param[in] iv either the \p iv passed to \ref sdc_encrypt or the internally generated iv
 * \param[in] iv_len length of \p iv
 *
 * Additional error codes
 * \return refer to \ref sdc_encrypt
 * \return SDC_NOT_SUPPORTED - no optimization implemented
 */
SDC_ARCH_API sdc_error_t sdc_arch_encrypt(sdc_session_t *session,
                                          const sdc_encrypt_decrypt_type_t *type,
                                          const sdc_encrypt_decrypt_desc_t *desc,
                                          const uint8_t *in_data,
                                          const size_t in_data_len,
                                          uint8_t *out_data,
                                          const size_t out_data_len,
                                          const uint8_t *iv, const size_t iv_len);

/**
 * \ingroup arch_encdec_op_aio
 * \brief Architecture specific to calculate upper limit for plain output
 *
 *  For decrypt and unwrap the common code needs to allocate memory to store the result.
 *  When using userspace padding and \ref sdc_intern_unpad this data needs to be large
 *  enough to hold data + padding.
 *
 * \param[in] session same as \p session of \ref sdc_decrypt
 * \param[in] type same as \p type of \ref sdc_decrypt
 * \param[in] desc - filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] in_len length of the cipher data to decrypt
 * \param[out] out_len_max upper limit of plain output
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INTERNAL_ERROR - this probably means that a new value has been added to sdc_padding_t without handling it here
 */
SDC_ARCH_API sdc_error_t sdc_arch_decrypt_get_max_out_len(sdc_session_t *session,
                                             const sdc_encrypt_decrypt_type_t *type,
                                             const sdc_encrypt_decrypt_desc_t *desc,
                                             const size_t in_len,
                                             size_t *out_len_max);

/**
 * \ingroup arch_encdec_op_aio
 * \brief Architecture specific function to execute an optimized
 * decrypt operation
 *
 * Architecture specific function to execute an optimized
 * decrypt operation instead of using \ref sdc_decrypt_init,
 * \ref sdc_decrypt_update, \ref sdc_decrypt_finalize.
 * In case the architecture does not provide the optimization this function
 * has to return \ref SDC_NOT_SUPPORTED
 * In this case the common part of SDC library will use the mandatory functions
 * \ref sdc_decrypt_init, \ref sdc_decrypt_update, \ref sdc_decrypt_finalize
 * for realization
 *
 * This function maps very closely to \ref sdc_decrypt as available to applications
 * In contrast to this function \p desc is provided and the type of \p out_data is a
 * simple pointer (memory is allocated in surrounding common wrapper).
 * Last but not least most NULL pointer checks have already been performed.
 *
 * Note: In case the decryption type needs padding to the block alignment the
 * input data and the output data length is aligned to the block alignment (and include padding)
 * Removing the bytes padded during encrypt is handled in the surrounding function.
 *
 * Please refer to \ref sdc_decrypt for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_decrypt
 * \param[in] type same as \p type of \ref sdc_decrypt
 * \param[in] desc - filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] in_data same as \p in_data of \ref sdc_decrypt
 * \param[in] in_data_len length of \p in_data
 * \param[out] out_data pointer to the output buffer (allocated by \ref sdc_decrypt)
 * \param[in,out] out_data_len length of \p out_data - input avail buffer - output used buffer
 * \param[in] iv same as \p iv of \ref sdc_decrypt
 * \param[in] iv_len same as \p iv_len of \ref sdc_decrypt
 *
 * Additional error codes
 * \return refer to \ref sdc_decrypt
 * \return SDC_NOT_SUPPORTED - no optimization implemented
 */
SDC_ARCH_API sdc_error_t sdc_arch_decrypt(sdc_session_t *session,
                                          const sdc_encrypt_decrypt_type_t *type,
                                          const sdc_encrypt_decrypt_desc_t *desc,
                                          const uint8_t *in_data,
                                          const size_t in_data_len,
                                          uint8_t *out_data,
                                          size_t *out_data_len,
                                          const uint8_t *iv, const size_t iv_len);

/**
 * \ingroup arch_enc_op_iuf
 * \brief Architecture specific to calculate upper limit for the output of update or finalize (encrypt)
 *
 * Calculate the maximum output length of a single update or finalize
 * operation when encrypting a message of the given \p type.\n
 * This value has to be the upper limit for the given length \p in_len
 * and all smaller length.\n
 * This value needs to be the upper limit, no matter if one or many update calls are used.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] in_len max length passed to any update call
 * \param[out] out_len_max upper limit of plain output
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INTERNAL_ERROR - this probably means that a new value has been added to sdc_padding_t without handling it here
 */
SDC_ARCH_API sdc_error_t sdc_arch_encrypt_get_update_len(sdc_session_t *session,
                                                         const sdc_encrypt_decrypt_type_t *type,
                                                         const sdc_encrypt_decrypt_desc_t *desc,
                                                         const size_t in_len,
                                                         size_t *out_len_max);

/**
 * \ingroup arch_dec_op_iuf
 * \brief Architecture specific to calculate upper limit for the output of update or finalize (decrypt)
 *
 * Calculate the maximum output length of a single update or finalize
 * operation when decrypting a message of the given \p type.\n
 * This value has to be the upper limit for the given length \p in_len
 * and all smaller length.\n
 * This value needs to be the upper limit, no matter if one or many update calls are used.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] in_len max length passed to any update call
 * \param[out] out_len_max upper limit of plain output
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INTERNAL_ERROR - this probably means that a new value has been added to sdc_padding_t without handling it here
 */
SDC_ARCH_API sdc_error_t sdc_arch_decrypt_get_update_len(sdc_session_t *session,
                                                         const sdc_encrypt_decrypt_type_t *type,
                                                         const sdc_encrypt_decrypt_desc_t *desc,
                                                         const size_t in_len,
                                                         size_t *out_len_max);

/**
 * \ingroup arch_enc_op_iuf
 * \brief Architecture specific function to initialize init, update, finalize
 * encryption sequence
 *
 * This function maps very closely to \ref sdc_encrypt_init as available to applications
 * \p iv, \p iv_len have either been provided by application or automatically generated
 * in the common part.
 * \n
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code already checked that \p iv, \p iv_len are valid for the given \p type.\n
 * \n
 * Please refer to \ref sdc_encrypt_init for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_encrypt_init
 * \param[in] type same as \p type of \ref sdc_encrypt_init
 * \param[in] desc - filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] iv       pointer to the IV buffer
 * \param[in] iv_len   length of the IV\n
 *
 * Additional error codes
 * \return refer to \ref sdc_encrypt_init
 * \return SDC_NOT_SUPPORTED - non-supported algorithm/mode
 */
SDC_ARCH_API sdc_error_t sdc_arch_encrypt_init(sdc_session_t *session,
                                               const sdc_encrypt_decrypt_type_t *type,
                                               const sdc_encrypt_decrypt_desc_t *desc,
                                               const uint8_t *iv, size_t iv_len);

/**
 * \ingroup arch_dec_op_iuf
 * \brief Architecture specific function to initialize init, update, finalize
 * decryption sequence
 *
 * This function maps very closely to \ref sdc_decrypt_init as available to applications
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code already checked that \p iv, \p iv_len are valid for the given \p type.\n
 * \n
 * Please refer to \ref sdc_decrypt_init for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_decrypt_init
 * \param[in] type same as \p type of \ref sdc_decrypt_init
 * \param[in] desc - filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] iv same as \p iv of \ref sdc_decrypt_init
 * \param[in] iv_len length of \p iv
 *
 * Additional error codes
 * \return refer to \ref sdc_decrypt_init
 * \return SDC_NOT_SUPPORTED - non-supported algorithm/mode
 */
SDC_ARCH_API sdc_error_t sdc_arch_decrypt_init(sdc_session_t *session,
                                               const sdc_encrypt_decrypt_type_t *type,
                                               const sdc_encrypt_decrypt_desc_t *desc,
                                               const uint8_t *iv, size_t iv_len);

/**
 * \ingroup arch_enc_op_iuf
 * \brief Architecture specific function to encrypt data using init, update, finalize
 *
 * This function maps very closely to \ref sdc_encrypt_update as available to applications.
 * The function needs to internally verify that the output buffer provided by
 * application needs to be sufficient large to handle the provided input.
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code already checked that \p in_data is not NULL
 * Note: The common code already checked that \p out_len and \p out_data_len are not NULL.\n
 * \n
 * The common code will guarantee that this function is only called with
 * 0 < \p in_data_len <= \p desc->data.max_chunk_len.\n
 * In case \p desc->data.chunk_len_aligned is true in_data_len is furthermore
 * aligned to desc->data.block_len.\n
 * In this case handling unaligned data length is handled in the common code.
 * Therefore a single \ref sdc_encrypt_update call by application can result in
 * multiple \ref sdc_arch_encrypt_update calls.
 * \n
 * Please refer to \ref sdc_encrypt_update for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_encrypt_update
 * \param[in] type same as \p type of \ref sdc_encrypt_update
 * \param[in] desc - filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] in_data same as \p in_data of \ref sdc_encrypt_update
 * \param[in] in_data_len same as \p in_len of \ref sdc_encrypt_update
 * \param[out] out_data pointer to the output buffer.
 * \param[in,out] out_data_len  pointer to the output buffer length\n
 *                              input: available length / output: used length\n
 *
 * Additional error codes
 * \return refer to \ref sdc_encrypt_update
 * \return SDC_NOT_SUPPORTED - non-supported algorithm/mode
 */
SDC_ARCH_API sdc_error_t sdc_arch_encrypt_update(sdc_session_t *session,
                                                 const sdc_encrypt_decrypt_type_t *type,
                                                 const sdc_encrypt_decrypt_desc_t *desc,
                                                 const uint8_t *in_data, const size_t in_data_len,
                                                 uint8_t *out_data, size_t *out_data_len);

/**
 * \ingroup arch_dec_op_iuf
 * \brief Architecture specific function to decrypt data using init, update, finalize
 *
 * This function maps very closely to \ref sdc_decrypt_update as available to applications.
 * The function needs to internally verify that the output buffer provided by
 * application needs to be sufficient large to handle the provided input.
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code already checked that \p in_data is not NULL
 * Note: The common code already checked that \p out_len and \p out_data_len are not NULL.\n
 * \n
 * The common code will guarantee that this function is only called with
 * 0 < \p in_data_len <= \p desc->data.max_chunk_len.\n
 * In case \p desc->data.chunk_len_aligned is true in_data_len is furthermore
 * aligned to desc->data.block_len.\n
 * In this case handling unaligned data length is handled in the common code.
 * Therefore a single \ref sdc_decrypt_update call by application can result in
 * multiple \ref sdc_arch_decrypt_update calls.
 * \n
 * Please refer to \ref sdc_decrypt_update for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_decrypt_update
 * \param[in] type same as \p type of \ref sdc_decrypt_update
 * \param[in] desc - filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] in_data same as \p in_data of \ref sdc_decrypt_update
 * \param[in] in_data_len same as \p in_len of \ref sdc_decrypt_update
 * \param[out] out_data pointer to the output buffer.
 * \param[in,out] out_data_len  pointer to the output buffer length\n
 *                              input: available length / output: used length\n
 *
 * Additional error codes
 * \return refer to \ref sdc_decrypt_update
 * \return SDC_NOT_SUPPORTED - non-supported algorithm/mode
 */
SDC_ARCH_API sdc_error_t sdc_arch_decrypt_update(sdc_session_t *session,
                                                 const sdc_encrypt_decrypt_type_t *type,
                                                 const sdc_encrypt_decrypt_desc_t *desc,
                                                 const uint8_t *in_data, const size_t in_data_len,
                                                 uint8_t *out_data, size_t *out_data_len);

/**
 * \ingroup arch_enc_op_iuf
 * \brief Architecture specific function to finalize an encrypt init, update, finalize sequence
 *
 * This function maps very closely to \ref sdc_encrypt_finalize as available to applications.
 * The function needs to internally verify that the output buffer provided by
 * application needs to be sufficient large to handle the provided input.
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code assures that \p in_data is not NULL if \p in_data_len != 0
 * Note: The common code already checked that \p out_len and \p out_data_len are not NULL.\n
 * \n
 * During update the common code will guarantee that \ref sdc_encrypt_update is only
 * called with 0 < \p in_data_len <= \p desc->data.max_chunk_len and
 * \p in_data_len fitting to \p desc->data.chunk_len_aligned.\n
 * The potential remaining unaligned part is passed by common code when
 * calling \ref sdc_arch_encrypt_finalize.\n
 * \n
 * If padding != SDC_PADDING_INTERNAL the padding will be handled by the
 * common code\n
 * If padding == SDC_PADDING_INTERNAL or SDC_PADDING_NO :
 * 0 <= \p in_data <= desc->data.block_len\n (0 if \p chunk_len_aligned == false)\n
 * Else \p in_data == desc->data.block_len\n
 *
 * Please refer to \ref sdc_encrypt_finalize for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_encrypt_finalize
 * \param[in] type same as \p type of \ref sdc_encrypt_finalize
 * \param[in] desc - filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] in_data potential unaligned data not handled by previous \ref sdc_arch_encrypt_update
 * \param[in] in_data_len length of \p in_data - 0 <= \p in_data <= desc->data.block_len
 * \param[out] out_data pointer to the output buffer.
 * \param[in,out] out_data_len  pointer to the output buffer length\n
 *                              input: available length / output: used length\n
 *
 * Additional error codes
 * \return refer to \ref sdc_decrypt_finalize
 * \return SDC_NOT_SUPPORTED - non-supported algorithm/mode
 */
SDC_ARCH_API sdc_error_t sdc_arch_encrypt_finalize(sdc_session_t *session,
                                                   const sdc_encrypt_decrypt_type_t *type,
                                                   const sdc_encrypt_decrypt_desc_t *desc,
                                                   const uint8_t *in_data, const size_t in_data_len,
                                                   uint8_t *out_data, size_t *out_data_len);

/**
 * \ingroup arch_dec_op_iuf
 * \brief Architecture specific function to finalize an decrypt init, update, finalize sequence
 *
 * This function maps very closely to \ref sdc_decrypt_finalize as available to applications.
 * The function needs to internally verify that the output buffer provided by
 * application needs to be sufficient large to handle the provided input.
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code assures that \p in_data is not NULL if \p in_data_len != 0
 * Note: The common code already checked that \p out_len and \p out_data_len are not NULL.\n
 * \n
 * During update the common code will guarantee that \ref sdc_decrypt_update is only
 * called with 0 < \p in_data_len <= \p desc->data.max_chunk_len and
 * \p in_data_len fitting to \p desc->data.chunk_len_aligned.\n
 * The potential remaining unaligned part is passed by common code when
 * calling \ref sdc_arch_decrypt_finalize.\n
 * \n
 * If padding != SDC_PADDING_INTERNAL removing the padding is done in the common code
 * after this function returned successfully.\n
 * If padding == SDC_PADDING_INTERNAL or SDC_PADDING_NO :
 * 0 <= \p in_data <= desc->data.block_len\n (0 if \p chunk_len_aligned == false)\n
 * Else \p in_data == desc->data.block_len\n*
 *
 * Please refer to \ref sdc_decrypt_finalize for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_decrypt_finalize
 * \param[in] type same as \p type of \ref sdc_decrypt_finalize
 * \param[in] desc - filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] in_data potential unaligned data not handled by previous \ref sdc_arch_decrypt_update
 * \param[in] in_data_len length of \p in_data - 0 <= \p in_data <= desc->data.block_len
 * \param[out] out_data pointer to the output buffer.
 * \param[in,out] out_data_len  pointer to the output buffer length\n
 *                              input: available length / output: used length\n
 *
 * Additional error codes
 * \return refer to \ref sdc_decrypt_finalize
 * \return SDC_NOT_SUPPORTED - non-supported algorithm/mode
 */
SDC_ARCH_API sdc_error_t sdc_arch_decrypt_finalize(sdc_session_t *session,
                                                   const sdc_encrypt_decrypt_type_t *type,
                                                   const sdc_encrypt_decrypt_desc_t *desc,
                                                   const uint8_t *in_data, const size_t in_data_len,
                                                   uint8_t *out_data, size_t *out_data_len);


/**
 * \ingroup arch_signverify
 * \defgroup arch_signverify_type Type
 * Type related functions,types etc.
 */

/**
 * \struct sdc_sign_verify_type
 * \ingroup arch_signverify_type
 * \brief Architecture specific struct to store the currently selected type for sign/verify.
 *
 * This type needs to store which \ref sdc_sign_verify_alg_t and \ref sdc_sign_verify_hash_t
 * has been selected by the user.
 * Furthermore it might be used to store the architecture specific representation of these
 * values or any other information related to the selected type.
 */

/**
 * \ingroup arch_signverify_type
 * \brief Return the default type for sign/verify in this architecture
 *
 * Please refer to \ref sdc_sign_verify_get_default for details.
 */
SDC_ARCH_API const sdc_sign_verify_type_t *sdc_arch_sign_verify_get_default(void);

/**
 * \ingroup arch_signverify_type
 * \brief Architecture specific allocation of \ref sdc_sign_verify_type_t
 *
 * Please refer to \ref sdc_sign_verify_type_alloc for details.
 * The architecture independent wrapper will assure that \p type != NULL.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_sign_verify_type_alloc(sdc_sign_verify_type_t **type);

/**
 * \ingroup arch_signverify_type
 * \brief Architecture specific deallocation of \ref sdc_sign_verify_type_t
 *
 * Please refer to \ref sdc_sign_verify_type_free for details.
 * The architecture independent wrapper will assure that \p type != NULL.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_sign_verify_type_free(sdc_sign_verify_type_t *type);

/**
 * \ingroup arch_signverify_type
 * \brief Architecture specific function to set \ref sdc_sign_verify_alg_t
 *
 * Please refer to \ref sdc_sign_verify_type_set_alg for details.
 * The architecture independent wrapper will assure that \p type != NULL and
 * \p alg in the range of \ref SDC_SIGNVER_ALG_FIRST and \ref SDC_SIGNVER_ALG_END - 1.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_sign_verify_type_set_alg(sdc_sign_verify_type_t *type, sdc_sign_verify_alg_t alg);

/**
 * \ingroup arch_signverify_type
 * \brief Architecture specific function to set \ref sdc_sign_verify_hash_t
 *
 * Please refer to \ref sdc_sign_verify_type_set_hash for details.
 * The architecture independent wrapper will assure that \p type != NULL and
 * \p hash in the range of \ref SDC_SIGNVER_HASH_FIRST and \ref SDC_SIGNVER_HASH_END - 1.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_sign_verify_type_set_hash(sdc_sign_verify_type_t *type, sdc_sign_verify_hash_t hash);

/**
 * \ingroup arch_signverify_type
 * \brief Architecture specific function to set additional options for the type
 *
 * Please refer to \ref sdc_sign_verify_type_set_opt_bmsk for details.
 * The architecture independent wrapper will assure that \p type != NULL and
 * \p opt_bmsk has no completely invalid bits set (no SDC_SIGNVER_OPT_*)
 */
SDC_ARCH_API sdc_error_t sdc_arch_sign_verify_type_set_opt_bmsk(sdc_sign_verify_type_t *type, uint64_t opt_bmsk);

/**
 * \ingroup arch_signverify_type
 * \brief Architecture specific function to read \ref sdc_sign_verify_alg_t from \ref sdc_sign_verify_type_t
 *
 * Please refer to \ref sdc_sign_verify_type_get_alg for details.
 * The architecture independent wrapper will assure that \p type != NULL and \p alg != NULL.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_sign_verify_type_get_alg(const sdc_sign_verify_type_t *type, sdc_sign_verify_alg_t *alg);

/**
 * \ingroup arch_signverify_type
 * \brief Architecture specific function to read \ref sdc_sign_verify_hash_t from \ref sdc_sign_verify_type_t
 *
 * Please refer to \ref sdc_sign_verify_type_get_hash for details.
 * The architecture independent wrapper will assure that \p type != NULL and \p hash != NULL.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_sign_verify_type_get_hash(const sdc_sign_verify_type_t *type, sdc_sign_verify_hash_t *hash);

/**
 * \ingroup arch_signverify_type
 * \brief Architecture specific function to read additional options from \ref sdc_sign_verify_type_t
 *
 * Please refer to \ref sdc_sign_verify_type_get_opt_bmsk for details.
 * The architecture independent wrapper will assure that \p type != NULL and \p opt_bmsk != NULL.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_sign_verify_type_get_opt_bmsk(const sdc_sign_verify_type_t *type, uint64_t *opt_bmsk);

/**
 * \ingroup arch_signverify_type
 * \brief Architecture specific function to fill \ref sdc_key_desc_t according to
 * the specified \ref sdc_sign_verify_type_t.
 *
 * This function needs to be architecture specific as not every architecture might support all potential
 * parameter ranges as defined by the format(specified by \ref sdc_sign_verify_type_t).
 *
 * The architecture independent wrapper will assure that \p type != NULL and \p key_desc != NULL.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_sign_verify_key_desc_fill (
    const sdc_sign_verify_type_t *type,
    sdc_key_desc_t *key_desc);

/**
 * \ingroup arch_signverify_type
 * \brief Architecture specific function to fill \ref sdc_sign_verify_desc_t according to
 * the specified \ref sdc_sign_verify_type_t.
 *
 * This function needs to be architecture specific as not every architecture might support all potential
 * parameter ranges as defined by the format(specified by \ref sdc_sign_verify_type_t).
 *
 * The architecture independent wrapper will assure that \p type != NULL and \p arg_desc != NULL.
 *
 * Note: session might be NULL. // TODO : get rid of this
 * In case the session is required to fill the descriptor (e.g. tag length depends on
 * loaded key length) the function needs to return SDC_SESSION_INVALID;
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_sign_verify_desc_fill (
    sdc_session_t *session,
    const sdc_sign_verify_type_t *type,
    sdc_sign_verify_desc_t *desc);

/**
 * \ingroup arch_signverify
 * \defgroup arch_signverify_op Operations
 * Operation related functions etc.
 */

/**
 * \ingroup arch_signverify_op
 * \defgroup arch_signverify_op_iuf Init,Update,Finalize
 * Init, Update, Finalize API for sign/verify
 */

/**
 * \ingroup arch_signverify_op_iuf
 * \defgroup arch_sign_op_iuf Init,Update,Finalize (Sign)
 * Perform sign operation using Init, Update, Finalize
 */

/**
 * \ingroup arch_signverify_op_iuf
 * \defgroup arch_verify_op_iuf Init,Update,Finalize ((Verify)
 * Perform verify operation using Init, Update, Finalize
 */

/**
 * \ingroup arch_signverify_op
 * \defgroup arch_signverify_op_aio All in one
 * Perform operation on the complete content in a single shot
 */

/**
 * \ingroup arch_signverify_op_aio
 * \brief Architecture specific function to execute an optimized
 * sign operation
 *
 * Architecture specific function to execute an optimized
 * sign operation instead of using \ref sdc_sign_init,
 * \ref sdc_sign_update, \ref sdc_sign_finalize.
 * In case the architecture does not provide the optimization this function
 * has to return \ref SDC_NOT_SUPPORTED
 * In this case the common part of SDC library will use the mandatory functions
 * \ref sdc_sign_init, \ref sdc_sign_update, \ref sdc_sign_finalize
 * for realization
 *
 * This function maps very closely to \ref sdc_sign as available to applications
 * In contrast to this function \p desc is provided and the type of \p tag_out_data is
 * a simple pointer (memory is allocated in surrounding common wrapper).
 *
 * Last but not least most NULL pointer checks have already been performed.
 *
 * Note: Adding padding needs to be done inside to allow optimization
 *
 * Please refer to \ref sdc_sign for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_sign
 * \param[in] type same as \p type of \ref sdc_sign
 * \param[in] desc - filled \ref sdc_sign_verify_desc_t descriptor
 * \param[in] in_data same as \p in_data of \ref sdc_sign
 * \param[in] in_data_len same as \p in_len of \ref sdc_sign
 * \param[in] iv either the \p iv passed to \ref sdc_sign or the internally generated iv
 * \param[in] iv_len length of \p iv
 * \param[out] tag_out_data pointer to the tag (mac or signature) buffer (allocated by \ref sdc_sign)
 * \param[in] tag_out_len length of \p tag_out
 *
 * Additional error codes
 * \return refer to \ref sdc_sign
 * \return SDC_NOT_SUPPORTED - no optimization implemented
 */
SDC_ARCH_API sdc_error_t sdc_arch_sign(sdc_session_t *session,
                                       const sdc_sign_verify_type_t *type,
                                       const sdc_sign_verify_desc_t *desc,
                                       const uint8_t *in_data,
                                       const size_t in_data_len,
                                       const uint8_t *iv, const size_t iv_len,
                                       uint8_t *tag_out_data, const size_t tag_out_len);

/**
 * \ingroup arch_sign_op_iuf
 * \brief Architecture specific function to initialize init, update, finalize
 * sign sequence
 *
 * This function maps very closely to \ref sdc_sign_init as available to applications
 * \p iv, \p iv_len have either been provided by application or automatically generated
 * in the common part.
 * \n
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code already checked that \p iv, \p iv_len are valid for the given \p type.\n
 * \n
 * Please refer to \ref sdc_sign_init for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_sign_init
 * \param[in] type same as \p type of \ref sdc_sign_init
 * \param[in] desc - filled \ref sdc_sign_verify_desc_t descriptor
 * \param[in] iv       pointer to the IV buffer
 * \param[in] iv_len   length of the IV\n
 *
 * Additional error codes
 * \return refer to \ref sdc_sign_init
 * \return SDC_NOT_SUPPORTED - non-supported algorithm/mode
 */
SDC_ARCH_API sdc_error_t sdc_arch_sign_init(sdc_session_t *session,
                                            const sdc_sign_verify_type_t *type,
                                            const sdc_sign_verify_desc_t *desc,
                                            const uint8_t *iv, size_t iv_len);

/**
 * \ingroup arch_sign_op_iuf
 * \brief Architecture specific function to sign data using init, update, finalize
 *
 * This function maps very closely to \ref sdc_sign_update as available to applications.
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code already checked that \p in_data is not NULL. \n
 * The common code will guarantee that this function is only called with
 * 0 < \p in_data_len <= \p desc->data.max_chunk_len.\n
 * In case \p desc->data.chunk_len_aligned is true in_data_len is furthermore
 * aligned to desc->data.block_len.\n
 * In this case handling unaligned data length is handled in the common code.
 * Therefore a single \ref sdc_sign_update call by application can result in
 * multiple \ref sdc_arch_sign_update calls.
 * \n
 * Please refer to \ref sdc_sign_update for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_sign_update
 * \param[in] type same as \p type of \ref sdc_sign_update
 * \param[in] desc - filled \ref sdc_sign_verify_desc_t descriptor
 * \param[in] in_data same as \p in_data of \ref sdc_sign_update
 * \param[in] in_data_len same as \p in_len of \ref sdc_sign_update
 *
 * Additional error codes
 * \return refer to \ref sdc_sign_update
 * \return SDC_NOT_SUPPORTED - non-supported algorithm/mode
 */
SDC_ARCH_API sdc_error_t sdc_arch_sign_update(sdc_session_t *session, const sdc_sign_verify_type_t *type,
                                              const sdc_sign_verify_desc_t *desc, const uint8_t *in_data,
                                              const size_t in_data_len);

/**
 * \ingroup arch_sign_op_iuf
 * \brief Architecture specific function to finalize an sign init, update, finalize sequence
 *
 * This function maps very closely to \ref sdc_sign_finalize as available to applications.
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code assures that \p in_data is not NULL if \p in_data_len != 0
 * Note: The common code already checked that \p tag_data_len is not zero and \p tag_data is not NULL.\n
 * \n
 * During update the common code will guarantee that \ref sdc_sign_update is only
 * called with 0 < \p in_data_len <= \p desc->data.max_chunk_len and
 * \p in_data_len fitting to \p desc->data.chunk_len_aligned.\n
 * The potential remaining unaligned part is passed by common code when
 * calling \ref sdc_arch_sign_finalize.\n
 * \n
 * If padding != SDC_PADDING_INTERNAL the padding will be handled by the
 * common code\n
 * If padding == SDC_PADDING_INTERNAL or SDC_PADDING_NO :
 * 0 <= \p in_data <= desc->data.block_len\n (0 if \p chunk_len_aligned == false)\n
 * Else \p in_data == desc->data.block_len\n
 *
 * Please refer to \ref sdc_sign_finalize for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_sign_finalize
 * \param[in] type same as \p type of \ref sdc_sign_finalize
 * \param[in] desc - filled \ref sdc_sign_verify_desc_t descriptor
 * \param[in] in_data potential unaligned data not handled by previous \ref sdc_arch_sign_update
 * \param[in] in_data_len length of \p in_data - 0 <= \p in_data <= desc->data.block_len
 * \param[out] tag_data pointer to the output buffer.
 * \param[in] tag_data_len length of output buffer.\n
 *
 * Additional error codes
 * \return refer to \ref sdc_sign_finalize
 * \return SDC_NOT_SUPPORTED - non-supported algorithm/mode
 */
SDC_ARCH_API sdc_error_t sdc_arch_sign_finalize(sdc_session_t *session,
                                                const sdc_sign_verify_type_t *type,
                                                const sdc_sign_verify_desc_t *desc,
                                                const uint8_t *in_data, const size_t in_data_len,
                                                uint8_t *tag_data, size_t tag_data_len);

/**
 * \ingroup arch_signverify_op_aio
 * \brief Architecture specific function to execute an optimized
 * verify operation
 *
 * Architecture specific function to execute an optimized
 * verify operation instead of using \ref sdc_verify_init,
 * \ref sdc_verify_update, \ref sdc_verify_finalize.
 * In case the architecture does not provide the optimization this function
 * has to return \ref SDC_NOT_SUPPORTED
 * In this case the common part of SDC library will use the mandatory functions
 * \ref sdc_verify_init, \ref sdc_verify_update, \ref sdc_verify_finalize
 * for realization
 *
 * This function maps very closely to \ref sdc_verify as available to applications
 * In contrast to this function \p desc is provided.
 *
 * Last but not least most NULL pointer checks have already been performed.
 *
 * Note: Adding padding needs to be done inside to allow optimization
 *
 * Please refer to \ref sdc_verify for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_verify
 * \param[in] type same as \p type of \ref sdc_verify
 * \param[in] desc - filled \ref sdc_sign_verify_desc_t descriptor
 * \param[in] in_data same as \p in_data of \ref sdc_verify
 * \param[in] in_data_len same as \p in_len of \ref sdc_verify
 * \param[in] iv same as \p iv of \ref sdc_verify
 * \param[in] iv_len same as \p iv_len of \ref sdc_verify
 * \param[in] tag_in_data (mac or signature) same as \p tag_data \ref sdc_verify
 * \param[in] tag_in_len same as \p tag_len of \ref sdc_verify
 *
 * Additional error codes
 * \return refer to \ref sdc_verify
 * \return SDC_NOT_SUPPORTED - no optimization implemented
 */

SDC_ARCH_API sdc_error_t sdc_arch_verify(sdc_session_t *session,
                                         const sdc_sign_verify_type_t *type,
                                         const sdc_sign_verify_desc_t *desc,
                                         const uint8_t *in_data,
                                         const size_t in_data_len,
                                         const uint8_t *iv, const size_t iv_len,
                                         const uint8_t *tag_in_data, const size_t tag_in_len);

/**
 * \ingroup arch_verify_op_iuf
 * \brief Architecture specific function to initialize init, update, finalize
 * verify sequence
 *
 * This function maps very closely to \ref sdc_verify_init as available to applications
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code already checked that \p iv, \p iv_len are valid for the given \p type.\n
 * \n
 * Please refer to \ref sdc_verify_init for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_verify_init
 * \param[in] type same as \p type of \ref sdc_verify_init
 * \param[in] desc - filled \ref sdc_sign_verify_desc_t descriptor
 * \param[in] iv same as \p iv of \ref sdc_verify_init
 * \param[in] iv_len length of \p iv
 *
 * Additional error codes
 * \return refer to \ref sdc_verify_init
 * \return SDC_NOT_SUPPORTED - non-supported algorithm/mode
 */
SDC_ARCH_API sdc_error_t sdc_arch_verify_init(sdc_session_t *session,
                                              const sdc_sign_verify_type_t *type,
                                              const sdc_sign_verify_desc_t *desc,
                                              const uint8_t *iv, size_t iv_len);

/**
 * \ingroup arch_verify_op_iuf
 * \brief Architecture specific function to verify data using init, update, finalize
 *
 * This function maps very closely to \ref sdc_verify_update as available to applications.
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code already checked that \p in_data is not NULL. \n
 * The common code will guarantee that this function is only called with
 * 0 < \p in_data_len <= \p desc->data.max_chunk_len.\n
 * In case \p desc->data.chunk_len_aligned is true in_data_len is furthermore
 * aligned to desc->data.block_len.\n
 * In this case handling unaligned data length is handled in the common code.
 * Therefore a single \ref sdc_verify_update call by application can result in
 * multiple \ref sdc_arch_verify_update calls.
 * \n
 * Please refer to \ref sdc_verify_update for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_verify_update
 * \param[in] type same as \p type of \ref sdc_verify_update
 * \param[in] desc - filled \ref sdc_sign_verify_desc_t descriptor
 * \param[in] in_data same as \p in_data of \ref sdc_verify_update
 * \param[in] in_data_len same as \p in_len of \ref sdc_verify_update
 *
 * Additional error codes
 * \return refer to \ref sdc_verify_update
 * \return SDC_NOT_SUPPORTED - non-supported algorithm/mode
 */
SDC_ARCH_API sdc_error_t sdc_arch_verify_update(sdc_session_t *session,
                                                const sdc_sign_verify_type_t *type,
                                                const sdc_sign_verify_desc_t *desc,
                                                const uint8_t *in_data, const size_t in_data_len);

/**
 * \ingroup arch_verify_op_iuf
 * \brief Architecture specific function to finalize an verify init, update, finalize sequence
 *
 * This function maps very closely to \ref sdc_verify_finalize as available to applications.
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code assures that \p in_data is not NULL if \p in_data_len != 0
 * Note: The common code already checked that \p tag_data_len is not zero and \p tag_data is not NULL.\n
 * \n
 * During update the common code will guarantee that \ref sdc_verify_update is only
 * called with 0 < \p in_data_len <= \p desc->data.max_chunk_len and
 * \p in_data_len fitting to \p desc->data.chunk_len_aligned.\n
 * The potential remaining unaligned part is passed by common code when
 * calling \ref sdc_arch_verify_finalize.\n
 * \n
 * If padding != SDC_PADDING_INTERNAL the padding will be handled by the
 * common code\n
 * If padding == SDC_PADDING_INTERNAL or SDC_PADDING_NO :
 * 0 <= \p in_data <= desc->data.block_len\n (0 if \p chunk_len_aligned == false)\n
 * Else \p in_data == desc->data.block_len\n
 *
 * Please refer to \ref sdc_verify_finalize for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_verify_finalize
 * \param[in] type same as \p type of \ref sdc_verify_finalize
 * \param[in] desc - filled \ref sdc_sign_verify_desc_t descriptor
 * \param[in] in_data potential unaligned data not handled by previous \ref sdc_arch_verify_update
 * \param[in] in_data_len length of \p in_data - 0 <= \p in_data <= desc->data.block_len
 * \param[in] tag_data pointer to the output buffer.
 * \param[in] tag_data_len length of output buffer.\n
 *
 * Additional error codes
 * \return refer to \ref sdc_verify_finalize
 * \return SDC_NOT_SUPPORTED - non-supported algorithm/mode
 */
SDC_ARCH_API sdc_error_t sdc_arch_verify_finalize(sdc_session_t *session,
                                                  const sdc_sign_verify_type_t *type,
                                                  const sdc_sign_verify_desc_t *desc,
                                                  const uint8_t *in_data, const size_t in_data_len,
                                                  const uint8_t *tag_data, size_t tag_data_len);

/**
 * \ingroup arch_digest
 * \defgroup arch_digest_type Type
 * Type related functions,types etc.
 */

/**
 * \struct sdc_dgst_type
 * \ingroup arch_digest_type
 * \brief Architecture specific struct to store the currently selected type for digest.
 *
 * This type needs to store which \ref sdc_dgst_hash_t has been selected by the user.
 * Furthermore it might be used to store the architecture specific representation of these
 * values or any other information related to the selected type.
 */

/**
 * \ingroup arch_digest_type
 * \brief Return the default type for digest in this architecture
 *
 * Please refer to \ref sdc_dgst_get_default for details.
 */

SDC_ARCH_API const sdc_dgst_type_t *sdc_arch_dgst_get_default(void);

/**
 * \ingroup arch_digest_type
 * \brief Architecture specific allocation of \ref sdc_dgst_type_t
 *
 * Please refer to \ref sdc_dgst_type_alloc for details.
 * The architecture independent wrapper will assure that \p type != NULL.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_dgst_type_alloc(sdc_dgst_type_t **type);

/**
 * \ingroup arch_digest_type
 * \brief Architecture specific deallocation of \ref sdc_dgst_type_t
 *
 * Please refer to \ref sdc_dgst_type_free for details.
 * The architecture independent wrapper will assure that \p type != NULL.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_dgst_type_free(sdc_dgst_type_t *type);

/**
 * \ingroup arch_digest_type
 * \brief Architecture specific function to set \ref sdc_dgst_hash_t
 *
 * Please refer to \ref sdc_dgst_type_set_hash for details.
 * The architecture independent wrapper will assure that \p type != NULL and
 * \p hash in the range of \ref SDC_DGST_HASH_FIRST and \ref SDC_DGST_HASH_END - 1.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_dgst_type_set_hash(sdc_dgst_type_t *type, sdc_dgst_hash_t hash);

/**
 * \ingroup arch_digest_type
 * \brief Architecture specific function to read \ref sdc_dgst_hash_t from \ref sdc_dgst_type_t
 *
 * Please refer to \ref sdc_dgst_type_get_hash for details.
 * The architecture independent wrapper will assure that \p type != NULL and \p hash != NULL.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_dgst_type_get_hash(const sdc_dgst_type_t *type, sdc_dgst_hash_t *hash);

/**
 * \ingroup arch_digest_type
 * \brief Architecture specific function to fill \ref sdc_dgst_desc_t according to
 * the specified \ref sdc_dgst_type_t.
 *
 * This function needs to be architecture specific as not every architecture might support all potential
 * parameter ranges as defined by the format(specified by \ref sdc_dgst_type_t).
 *
 * The common functions used for digest will check if the specified parameters are valid.
 * Furthermore these functions may align the data length to the block length if required.
 * In order to do so these function will use this function to fill \p desc with the information used to verify.
 *
 * The architecture independent wrapper will assure that \p type != NULL and \p desc != NULL.
 *
 */
SDC_ARCH_API sdc_error_t sdc_arch_dgst_desc_fill (const sdc_dgst_type_t *type, sdc_dgst_desc_t *desc);

/**
 * \ingroup arch_digest
 * \defgroup arch_digest_op Operations
 * Operation related functions etc.
 */

/**
 * \ingroup arch_digest_op
 * \defgroup arch_digest_op_iuf Init,Update,Finalize
 * Perform operation for the currently provided part using Init, Update, Finalize
 */

/** \ingroup arch_digest_op_iuf
 * \brief Architecture specific function to initialize init, update, finalize
 * digest sequence
 *
 * This function maps very closely to \ref sdc_dgst_init as available to applications. \n
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Please refer to \ref sdc_dgst_init for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_dgst_init
 * \param[in] type same as \p type of \ref sdc_dgst_init
 * \param[in] desc - filled \ref sdc_dgst_desc_t descriptor
 *
 * Additional error codes
 * \return refer to \ref sdc_dgst_init
 * \return SDC_NOT_SUPPORTED - non-supported algorithm/mode
 */
SDC_ARCH_API sdc_error_t sdc_arch_dgst_init(sdc_session_t *session,
                                            const sdc_dgst_type_t *type,
                                            const sdc_dgst_desc_t *desc);
/**
 * \ingroup arch_digest_op_iuf
 * \brief Architecture specific function to digest data using init, update, finalize
 *
 * This function maps very closely to \ref sdc_dgst_update as available to applications.
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code already checked that \p in_data is not NULL. \n
 * The common code will guarantee that this function is only called with
 * 0 < \p in_data_len <= \p desc->data.max_chunk_len.\n
 * In case \p desc->data.chunk_len_aligned is true in_data_len is furthermore
 * aligned to desc->data.block_len.\n
 * In this case handling unaligned data length is handled in the common code.
 * Therefore a single \ref sdc_dgst_update call by application can result in
 * multiple \ref sdc_arch_dgst_update calls.
 * \n
 * Please refer to \ref sdc_dgst_update for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_dgst_update
 * \param[in] type same as \p type of \ref sdc_dgst_update
 * \param[in] desc - filled \ref sdc_dgst_desc_t descriptor
 * \param[in] in_data same as \p in_data of \ref sdc_dgst_update
 * \param[in] in_data_len same as \p in_len of \ref sdc_dgst_update
 *
 * Additional error codes
 * \return refer to \ref sdc_dgst_update
 * \return SDC_NOT_SUPPORTED - non-supported algorithm/mode
 */
SDC_ARCH_API sdc_error_t sdc_arch_dgst_update(sdc_session_t *session,
                                              const sdc_dgst_type_t *type,
                                              const sdc_dgst_desc_t *desc,
                                              const uint8_t *in_data, const size_t in_data_len);
/**
 * \ingroup arch_digest_op_iuf
 * \brief Architecture specific function to finalize an digest init, update, finalize sequence
+ *
 * This function maps very closely to \ref sdc_dgst_finalize as available to applications.
 * Note: Null pointer checks of \p session, \p type and \p desc have been
 * performed in the common code.\n
 * Note: The common code assures that \p in_data is not NULL if \p in_data_len != 0
 * Note: The common code already checked that \p digest_data_len is not zero and \p digest_data is not NULL.\n
 * \n
 * During update the common code will guarantee that \ref sdc_dgst_update is only
 * called with 0 < \p in_data_len <= \p desc->data.max_chunk_len and
 * \p in_data_len fitting to \p desc->data.chunk_len_aligned.\n
 * The potential remaining unaligned part is passed by common code when
 * calling \ref sdc_arch_dgst_finalize.\n
 * \n
 * If padding != SDC_PADDING_INTERNAL the padding will be handled by the
 * common code\n
 * If padding == SDC_PADDING_INTERNAL or SDC_PADDING_NO :
 * 0 <= \p in_data <= desc->data.block_len\n (0 if \p chunk_len_aligned == false)\n
 * Else \p in_data == desc->data.block_len\n
 *
 * Please refer to \ref sdc_dgst_finalize for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_dgst_finalize
 * \param[in] type same as \p type of \ref sdc_dgst_finalize
 * \param[in] desc - filled \ref sdc_dgst_desc_t descriptor
 * \param[in] in_data potential unaligned data not handled by previous \ref sdc_arch_dgst_update
 * \param[in] in_data_len length of \p in_data - 0 <= \p in_data <= desc->data.block_len
 * \param[out] digest_data pointer to the output digest buffer.
 * \param[in] digest_data_len length of output digest buffer.\n
 *
 * Additional error codes
 * \return refer to \ref sdc_dgst_finalize
 * \return SDC_NOT_SUPPORTED - non-supported algorithm/mode
 */

SDC_ARCH_API sdc_error_t sdc_arch_dgst_finalize(sdc_session_t *session,
                                                const sdc_dgst_type_t *type,
                                                const sdc_dgst_desc_t *desc,
                                                const uint8_t *in_data, const size_t in_data_len,
                                                uint8_t *digest_data, size_t digest_data_len);

/**
 * \ingroup arch_digest_op
 * \defgroup arch_digest_op_aio All in one
 * Perform operation on the complete content in a single shot
 */

/**
 * \ingroup arch_digest_op_aio
 * \brief Architecture specific function to execute an optimized
 * dgst operation
 *
 * Architecture specific function to execute an optimized
 * dgst operation instead of using \ref sdc_dgst_init,
 * \ref sdc_dgst_update, \ref sdc_dgst_finalize.
 * In case the architecture does not provide the optimization this function
 * has to return \ref SDC_NOT_SUPPORTED
 * In this case the common part of SDC library will use the mandatory functions
 * \ref sdc_dgst_init, \ref sdc_dgst_update, \ref sdc_dgst_finalize
 * for realization
 *
 * This function maps very closely to \ref sdc_dgst as available to applications
 * In contrast to this function \p desc is provided and the type of \p digest_out_data is
 * a simple pointer (memory is allocated in surrounding common wrapper).
 *
 * Last but not least most NULL pointer checks have already been performed.
 *
 * Note: Adding padding needs to be done inside to allow optimization
 *
 * Please refer to \ref sdc_dgst for more information on parameters and error codes
 *
 * \param[in] session same as \p session of \ref sdc_dgst
 * \param[in] type same as \p type of \ref sdc_dgst
 * \param[in] desc - filled \ref sdc_dgst_desc_t descriptor
 * \param[in] in_data same as \p in_data of \ref sdc_dgst
 * \param[in] in_data_len same as \p in_len of \ref sdc_dgst
 * \param[out] digest_out_data pointer to the digest buffer (allocated by \ref sdc_dgst)
 * \param[in] digest_out_len length of \p tag_out
 *
 * Additional error codes
 * \return refer to \ref sdc_dgst
 * \return SDC_NOT_SUPPORTED - no optimization implemented
 */
SDC_ARCH_API sdc_error_t sdc_arch_dgst(sdc_session_t *session,
                                       const sdc_dgst_type_t *type,
                                       const sdc_dgst_desc_t *desc,
                                       const uint8_t *in_data,
                                       const size_t in_data_len,
                                       uint8_t *digest_out_data, const size_t digest_out_len);


/**
 * \ingroup arch_gen
 * \brief Architecture specific function to return the supported bitmasks for \p key_fmt
 *
 * Please refer to \ref sdc_key_len_bmsks
 * NULL pointer check of arch_keylen_bmsk has already been performed in the
 * wrapper
 * Furthermore it is already checked that key_fmt is a valid entry of \ref sdc_key_len_bmsk_t
 */
SDC_ARCH_API sdc_error_t sdc_arch_key_len_bmsks(sdc_key_fmt_t key_fmt, sdc_key_len_bmsk_t *arch_keylen_bmsk);

/**
 * \ingroup arch_gen
 * \brief Architecture specific function to verify that libSDC and kernel versions (if applicable) match
 *
 * Please refer to \ref sdc_session_unset_key for details on the parameters and return values.
 *
 * In case no kernel is involved it is save to return \ref SDC_OK.
 */
SDC_ARCH_API sdc_error_t sdc_arch_kernel_version_verify(char **version_info_str);

#endif
